#! /usr/bin/env julia
# ---------------------------------------------------------


# ---------------------------------------------------------
using Distributed
addprocs(Sys.CPU_THREADS - 1); # Add all remaining available CPU threads.
@info "Number of procs for parallel mapping ... " * string(nprocs())
@everywhere push!(LOAD_PATH, "./src")
@everywhere using project_routines
@everywhere using ProgressMeter

using CSV, FileIO
using DataFrames
using Parameters

using Random
using Statistics: mean, std, cor, I, kron, UpperTriangular, median
using Plots
using LaTeXStrings
using JLD2
using ProgressMeter
using Compat
using SortingAlgorithms
using ParallelDataTransfer
pgfplotsx(size = (1400, 900))
# ---------------------------------------------------------


# ---------------------------------------------------------
# logging
run(`mkdir -p log`)
using Logging, LoggingExtras, Dates

# clean up the files
map(x->rm(x, force=true), ["./log/table11_info.log.jl", "./log/table11_warn.log.jl", "./log/table11_full.log.jl"]);
# custom timestamps
const log_date_format = "yyyy-mm-dd HH:MM:SS"
timestamp_logger(logger) = TransformerLogger(logger) do log
  merge(log, (; message = "$(Dates.format(now(), log_date_format)) \n$(log.message)"))
end

# create the debugger
demux_logger = TeeLogger(
    MinLevelLogger(FileLogger("./log/table11_info.log.jl"), Logging.Info),
    MinLevelLogger(FileLogger("./log/table11_warn.log.jl"), Logging.Warn),
    MinLevelLogger(FileLogger("./log/table11_full.log.jl"), Logging.Debug),
    ConsoleLogger(stdout, Logging.Info),   # Common logger to be able to see info and warning in repl
) |> timestamp_logger |> global_logger;
# ---------------------------------------------------------


# ---------------------------------------------------------
## Load in the data
@info "Load the data"
calibration_parameters = CSV.read("./input/param_calibration.csv", DataFrame);
Lj = CSV.read("./input/Ljstar.csv", DataFrame).Lj;
PjCj = CSV.read("./input/PjCj.csv", DataFrame);
Tau = CSV.read("./input/Taustar.csv", DataFrame) |> Matrix;

dict_in = load("./tmp/simulation_dict.jld2");
# ---------------------------------------------------------


# ---------------------------------------------------------
@info "Parameters"
eta, psi, sigma, gamma, phi, years, dt, BURN_IN =  Vector(calibration_parameters[1, :]);
N = size(Tau, 1);
t = (0:dt:years-dt);

Pj_lambda = PjCj.Pj_lambda; 
Cj_lambda = PjCj.Cj_lambda;
lambdaj_pp_gamma0 = Pj_lambda ./ Cj_lambda .^ (-gamma);
lambdaj_pp_gamma = lambdaj_pp_gamma0 ./ sum(lambdaj_pp_gamma0);

calibration = parameters_TradeWar(eta, gamma, N, dt, phi,
    Lj, lambdaj_pp_gamma, Tau, 0.2, 1e-5);

## Generate the random state path
states = random_productivity_state(N, t, sigma, psi, years, dt, BURN_IN);
# ---------------------------------------------------------


# ---------------------------------------------------------
# Iterate on all pair (upper triangle of country-pair matrices)
# debugging with lower amount
N_sub = N
iter_counterfactuals = [];
[ i>j ? append!(iter_counterfactuals, [vec([i j])]) : 0 for i=1:N_sub, j=1:N_sub ];

sendto(workers(), states=states)
sendto(workers(), calibration=calibration)

GenOutputs = [];
GenOutputs = @showprogress pmap(iter_counterfactuals) do idx_iter
  res_tmp = bilateral_trade_deal_cf(idx_iter, states, calibration; 
                                    TauReductionFactor = 0.8);
end

CSV.write("./tmp/simulation_dict_table11.csv", GenOutputs) # just in case something fails
# ---------------------------------------------------------


# ---------------------------------------------------------
# Analyze the output and generate Table 11
Table_11 = bilateral_trade_deal_analyze(;
  dict_out_org = dict_in,
  BilateralDealsOutputs = GenOutputs,
);

@info Table_11
CSV.write("./output/Table_11.csv", Table_11)
# ---------------------------------------------------------


# ---------------------------------------------------------
t11 = CSV.read("./output/Table_11.csv", DataFrame)

table_header = LaTeXString(raw"""
% Table generated by table11.jl in julia ...

\begin{tabular}{lrcccc}
\toprule
""")

table_avg_effects = L"""
\multicolumn{6}{c}{Panel A: Average Effects} \\
\midrule
     &      &      & Factor & Systematic & Unshared \\
     &      & Volatility & Loading & Share & Risk \\
     & \multicolumn{1}{c}{Statistic} & $\sigma$ & $\beta$ & $R^2$ & $\rho$ \\
     
\cmidrule{2-6}\multirow{2}[1]{*}{Before} & \multicolumn{1}{c}{Mean} & 
     %$(t11[1,3])    & %$(t11[1,4])    & %$(t11[1,5])    & %$(t11[1,6]) \\
     & \multicolumn{1}{c}{Stdev} & 
     %$(t11[2,3])    & %$(t11[2,4])    & %$(t11[2,5])    & %$(t11[2,6]) \\
     &      &      &      &      &  \\
After & \multicolumn{1}{c}{Mean}  & 
  %$(t11[3,3])    & %$(t11[3,4])    & %$(t11[3,5])    & %$(t11[3,6]) \\
\quad Difference  &          & 
  %$(t11[4,3])    & %$(t11[4,4])    & %$(t11[4,5])    & %$(t11[4,6]) \\
\quad Fraction of  Stdev &          & 
  %$(t11[5,3])    & %$(t11[5,4])    & %$(t11[5,5])    & %$(t11[5,6]) \\
\midrule
"""

table_conditional_effects = L"""
\multicolumn{6}{c}{Panel B: Conditional Effects} \\
\midrule
     &      &      & Factor & Systematic & Unshared \\
     &      & Volatility & Loading & Share & Risk \\
\textit{Core} & \multicolumn{1}{c}{Statistic} & $\sigma$ & $\beta$ & $R^2$ & $\rho$ \\
\cmidrule{2-6}\multirow{2}[1]{*}{Before} & \multicolumn{1}{c}{Mean} & 
     %$(t11[6,3])    & %$(t11[6,4])    & %$(t11[6,5])    & %$(t11[6,6]) \\
     & \multicolumn{1}{c}{Stdev} & 
     %$(t11[7,3])    & %$(t11[7,4])    & %$(t11[7,5])    & %$(t11[7,6]) \\
     &      &      &      &      &  \\
After & \multicolumn{1}{c}{Mean}  & 
  %$(t11[8,3])    & %$(t11[8,4])    & %$(t11[8,5])    & %$(t11[8,6]) \\
\quad Difference &          & 
  %$(t11[9,3])    & %$(t11[9,4])    & %$(t11[9,5])    & %$(t11[9,6]) \\
\quad Fraction of Stdev&          & 
  %$(t11[10,3])    & %$(t11[10,4])    & %$(t11[10,5])    & %$(t11[10,6]) \\
     &      &      &      &      &  \\
\textit{Peripheral} &      &      &      &      &  \\
\cmidrule{2-6}\multirow{2}[1]{*}{Before} & \multicolumn{1}{c}{Mean} & 
  %$(t11[11,3])    & %$(t11[11,4])    & %$(t11[11,5])    & %$(t11[11,6]) \\
     & \multicolumn{1}{c}{Stdev} & 
     %$(t11[12,3])    & %$(t11[12,4])    & %$(t11[12,5])    & %$(t11[12,6]) \\
     &      &      &      &      &  \\
After & \multicolumn{1}{c}{Mean} & 
  %$(t11[13,3])    & %$(t11[13,4])    & %$(t11[13,5])    & %$(t11[13,6]) \\
\quad Difference &          & 
  %$(t11[14,3])    & %$(t11[14,4])    & %$(t11[14,5])    & %$(t11[14,6]) \\
\quad Fraction of Stdev &          & 
    %$(t11[15,3])    & %$(t11[15,4])    & %$(t11[15,5])    & %$(t11[15,6]) \\
\bottomrule
\end{tabular}%
"""

# Export result to .tex file
write("./output/table_11.tex", 
  table_header * table_avg_effects * table_conditional_effects);
# ---------------------------------------------------------


